<?php
/**
 * PropertyModifierReader.php 2020-06-12
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */
declare(strict_types=1);

namespace Gambio\Shop\Properties\ProductModifiers\Database\Readers;

use CI_DB_query_builder;
use Gambio\Shop\Language\ValueObjects\LanguageId;
use Gambio\Shop\Product\ValueObjects\ProductId;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOBuilderInterface;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOCollection;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOCollectionInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\Interfaces\ModifierReaderCompositeInterface;
use Gambio\Shop\Properties\ProductModifiers\Database\ValueObjects\PropertyGroupIdentifier;
use Gambio\Shop\Properties\ProductModifiers\Database\ValueObjects\PropertyModifierIdentifier;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitId;

/**
 * Class PropertyModifierReader
 *
 * @package Gambio\Shop\Properties\ProductModifiers\Database\Readers
 */
class PropertyModifierReader implements ModifierReaderCompositeInterface
{

    /**
     * @var ModifierDTOBuilderInterface
     */
    protected $builder;
    /**
     * @var CI_DB_query_builder
     */
    protected $queryBuilder;
    /**
     * @var bool
     */
    private $stockCheck;
    /**
     * @var bool
     */
    private $attributeStockCheck;
    /**
     * @var bool
     */
    private $stockAllowCheckout;


    /**
     * PropertyModifierReader constructor.
     *
     * @param CI_DB_query_builder $queryBuilder
     * @param ModifierDTOBuilderInterface $builder
     * @param bool $stockCheck
     * @param bool $attributeStockCheck
     * @param bool $stockAllowCheckout
     */
    public function __construct(
        CI_DB_query_builder $queryBuilder,
        ModifierDTOBuilderInterface $builder,
        bool $stockCheck,
        bool $attributeStockCheck,
        bool $stockAllowCheckout
    ) {
        $this->queryBuilder        = $queryBuilder;
        $this->builder             = $builder;
        $this->stockCheck          = $stockCheck;
        $this->attributeStockCheck = $attributeStockCheck;
        $this->stockAllowCheckout  = $stockAllowCheckout;
    }


    /**
     * @inheritDoc
     */
    public function getModifierByProduct(ProductId $id, LanguageId $languageId): ModifierDTOCollectionInterface
    {
        $result = new ModifierDTOCollection();

        $sql = "SELECT p.properties_id, p.display_type, pv.display_image, pvd.values_name, pvd.properties_values_id, pv.`value_price`
                    FROM products_properties_combis ppc
                        INNER JOIN products_properties_combis_values ppcv
                            ON ppc.products_properties_combis_id = ppcv.products_properties_combis_id
                        INNER JOIN properties_values_description pvd
                            ON pvd.properties_values_id = ppcv.properties_values_id AND pvd.language_id = {$languageId->value()}
                        INNER JOIN properties_values pv
                            ON pv.properties_values_id = ppcv.properties_values_id
                        INNER JOIN properties p
                            ON p.properties_id = pv.properties_id
                    WHERE ppc.products_id = {$id->value()}
                    GROUP BY p.properties_id, pvd.values_name, pvd.properties_values_id
                    ORDER BY ppc.sort_order";

        $data = $this->queryBuilder->query($sql)->result_array();

        foreach ($data as $item) {

            [$pricePrefix, $priceValueFloat] = $this->pricePrefixAndValueFromFloat((float)$item['value_price']);

            $result->addModifier($this->builder->withId(new PropertyModifierIdentifier((int)$item['properties_values_id']))
                ->withGroupId(new PropertyGroupIdentifier((int)$item['properties_id']))
                ->withType($item['display_type'])
                ->withName($item['values_name'])
                ->withPrice($priceValueFloat)
                ->withPricePrefix($pricePrefix)
                ->withImage($item['display_image'] ? 'product_images/property_images/' . $item['display_image'] : '')
                ->withSource('property')
                ->build());
        }

        return $result;
    }


    /**
     * @param float $priceValueFloat
     *
     * @return mixed[]
     */
    protected function pricePrefixAndValueFromFloat(float $priceValueFloat): array
    {
        $pricePrefix     = $priceValueFloat >= 0 ? '+ ' : '- ';
        $priceValueFloat = $pricePrefix === '- ' ? $priceValueFloat * -1 : $priceValueFloat;

        return [$pricePrefix, $priceValueFloat];
    }

    /**
     * @inheritDoc
     */
    public function getModifierBySellingUnit(SellingUnitId $id, LanguageId $languageId): ModifierDTOCollectionInterface
    {
        $result = new ModifierDTOCollection();
        $structure = $this->getProperties(
            $id->productId()->value(),
            $id->language()->value()
        );
        if(!count($structure['groups'])) return $result;

        $selectedValues = [];
        foreach ($id->modifiers() as $modifier) {
            if($modifier instanceof PropertyModifierIdentifier) {
                $selectedValues[] = $modifier->value();
            }
        }
        $optionsList = $structure['value'];
        if(count($selectedValues))
        {
            $invalidOptions = $this->getUnavailableCombinationsList(
                $id->productId()->value(),
                $id->language()->value(),
                array_keys($structure['groups'])
            );

            //remove all invalid options form the ValidOptionsList
            foreach ($selectedValues as $selectedPropertyValueId) {
                foreach ($invalidOptions as $combinationId => $propertyValueIdList) {
                    if (isset($propertyValueIdList[$selectedPropertyValueId])) {
                        unset($propertyValueIdList[$selectedPropertyValueId]);
                        foreach ($propertyValueIdList as $propertyValueId => $value) {
                            if (!in_array($propertyValueId, $selectedValues)) {
                                $optionsList[$propertyValueId]['selectable'] = $value;
                            }
                        }
                    }
                }
            }
        }

        foreach ($optionsList as $item) {

            [$pricePrefix, $priceValueFloat] = $this->pricePrefixAndValueFromFloat((float)$item['value_price']);

            $propertyId = new PropertyModifierIdentifier((int)$item['properties_values_id']);
            $selected   = $id->modifiers()->indexOf($propertyId) >= 0;

            $result->addModifier($this->builder->withId($propertyId)
                ->withGroupId(new PropertyGroupIdentifier((int)$item['properties_id']))
                ->withType($item['display_type'])
                ->withName($item['values_name'])
                ->withSelectable($this->isSelectable($item))
                ->withSelected($selected)
                ->withPrice($priceValueFloat)
                ->withPricePrefix($pricePrefix)
                ->withImage($item['display_image'] ? 'product_images/property_images/' . $item['display_image'] : '')
                ->withSource('property')
                ->build());
        }

        return $result;
    }

    /**
     * @param $item
     *
     * @return bool
     */
    protected function isSelectable($item): bool
    {
        return in_array((int)$item['selectable'], [1,2]) || $item['properties_dropdown_mode'] === '';
    }

    protected function getUnavailableCombinationsList($productId, $languageId, $groups)
    {

        $fieldsSql = '';
        foreach ($groups as $groupId) {
            $fieldsSql .= ", max(case when pv.properties_id = {$groupId} then  pvd.properties_values_id end) properties_values_id_{$groupId} \n";
        }

        $sql = <<<SQL
            SELECT 
                   ppc.products_properties_combis_id
                   $fieldsSql
            FROM products_properties_combis ppc
                     INNER JOIN products_properties_combis_values ppcv
                                ON ppc.products_properties_combis_id = ppcv.products_properties_combis_id
                     INNER JOIN properties_values_description pvd
                                ON pvd.properties_values_id = ppcv.properties_values_id AND pvd.language_id = $languageId
                     INNER JOIN properties_values pv
                                ON pv.properties_values_id = ppcv.properties_values_id
                     INNER JOIN properties p
                                ON p.properties_id = pv.properties_id
                     INNER JOIN  products
                                 ON products.products_id = ppc.products_id
            WHERE ppc.products_id = $productId
SQL;
        //quantity check rules
        if ($this->attributeStockCheck && $this->stockCheck) {
            $sql.= <<<SQL
             and not ( 
                 (coalesce(products.use_properties_combis_quantity,0) in(0,2) and ppc.combi_quantity >= products.gm_min_order)
              or ( products.use_properties_combis_quantity  = 1 and products.products_quantity >= products.gm_min_order)

            )
SQL;

        } elseif ($this->stockCheck && $this->attributeStockCheck === false) {
            $sql.= <<<SQL
                and not ((coalesce(products.use_properties_combis_quantity,0) in(0,1) and products.products_quantity >= products.gm_min_order) 
                    or ( products.use_properties_combis_quantity in(2) and ppc.combi_quantity >= products.gm_min_order)  
                   )
SQL;

        }

        $sql.= <<<SQL
            group by products.use_properties_combis_quantity, ppc.combi_quantity, products.gm_min_order, products.products_quantity,
                     ppc.products_properties_combis_id
SQL;
        $invalidOptions = [];
        $data = $this->queryBuilder->query($sql)->result_array();
        foreach ($data as $row) {
            $invalidOptions[(int)$row['products_properties_combis_id']] = [];
            foreach ($groups as $groupId) {
                $fieldName = "properties_values_id_{$groupId}";

                $invalidOptions[(int)$row['products_properties_combis_id']][(int)$row[$fieldName]] = 0;
            }
        }

        return $invalidOptions;

    }

    protected function getProperties(int $productId, int $languageId): array
    {
        $sql    = <<<SQL
                SELECT          pv.properties_id,
                                p.display_type,
                                pv.display_image,
                                ppcv.properties_values_id,
                                pvd.values_name,
                                pv.`value_price`,
                                products.properties_dropdown_mode,
                                max({$this->availabilityStatus()}) selectable
                FROM products_properties_combis ppc
                         INNER JOIN products_properties_combis_values ppcv
                                    ON ppc.products_properties_combis_id = ppcv.products_properties_combis_id
                         INNER JOIN properties_values_description pvd
                                    ON pvd.properties_values_id = ppcv.properties_values_id
                                        AND pvd.language_id = {$languageId}
                         INNER JOIN products
                                    ON products.products_id = ppc.products_id
                         INNER JOIN properties_values pv
                                    ON pv.properties_values_id = ppcv.properties_values_id
                         INNER JOIN properties p
                                    ON p.properties_id = pv.properties_id
                
                WHERE ppc.products_id = {$productId}
                GROUP BY pv.properties_id,
                         p.display_type,
                         pv.display_image,
                         ppcv.properties_values_id,
                         pvd.values_name,
                         pv.`value_price`,
                         products.properties_dropdown_mode
                ORDER BY pv.properties_id, ppcv.properties_values_id
SQL;
        $data   = $this->queryBuilder->query($sql)->result_array();
        $values = [];
        $groups = [];
        foreach ($data as $row) {
            if (!isset($groups[(int)$row['properties_id']])) {
                $groups[(int)$row['properties_id']] = [];
            }
            $groups[(int)$row['properties_id']][(int)$row['properties_values_id']] = true;
            $values[(int)$row['properties_values_id']]                             = $row;
        }

        return [
            'value'  => $values,
            'groups' => $groups
        ];
    }

    /**
     * @param SellingUnitId $id
     *
     * @return string
     */
    protected function availabilityStatus()
    {

        $noStockResult = $this->stockAllowCheckout ? 2 : 0;

        if($this->attributeStockCheck && $this->stockCheck){
            return <<<SQL
             case
                   when coalesce(products.use_properties_combis_quantity,0) in(0,2) 
                        and ppc.combi_quantity >= products.gm_min_order then 1
                        
                    when coalesce(products.use_properties_combis_quantity,0) in(1) 
                        and products.products_quantity >= products.gm_min_order 
                        then 1
                   else $noStockResult
             end
SQL;
        }
        elseif($this->stockCheck && $this->attributeStockCheck === false)
        {
            return <<<SQL
                case
                    when coalesce(products.use_properties_combis_quantity,0) in(0,1) 
                        and products.products_quantity >= products.gm_min_order 
                        then 1
                    when coalesce(products.use_properties_combis_quantity,0) in(2) 
                        and ppc.combi_quantity >= products.gm_min_order  then 1
                    else $noStockResult
                end
SQL;
        } else {
            return '1';
        }

    }

}